from fastapi import FastAPI, Depends, HTTPException
from sqlalchemy.ext.asyncio import AsyncSession
from sqlalchemy import select, update
from app.database import engine, Base, get_db
from app.models import Match, ModelMetrics
from app.schemas import MatchCreate, MatchResponse, PredictionRequest, MatchPrediction
from app.analysis import FootballAnalyzer
from datetime import datetime, timedelta
import asyncio

app = FastAPI(title="Virtual Football Predictor")

# Startup Event to Create Tables
@app.on_event("startup")
async def startup():
    async with engine.begin() as conn:
        # In production, use Alembic for migrations
        await conn.run_sync(Base.metadata.create_all)

@app.post("/matches/", response_model=MatchResponse)
async def create_match(match: MatchCreate, db: AsyncSession = Depends(get_db)):
    """Add a new match result to the database."""
    db_match = Match(**match.dict())
    if not db_match.data_jogo:
        db_match.data_jogo = datetime.now()
    db.add(db_match)
    await db.commit()
    await db.refresh(db_match)
    return db_match

@app.get("/predict/", response_model=MatchPrediction)
async def predict_match(time_casa: str, time_fora: str, db: AsyncSession = Depends(get_db)):
    """Analyze and predict the outcome of a match."""
    analyzer = FootballAnalyzer(db)
    # Check if teams exist (basic check)
    # In a real scenario, we'd handle unknown teams more gracefully
    
    prediction = await analyzer.analyze_match(time_casa, time_fora)
    return prediction

from typing import List

@app.post("/matches/bulk-optimize/")
async def create_matches_and_optimize(matches: List[MatchCreate], db: AsyncSession = Depends(get_db)):
    """
    Insert multiple match results and trigger optimization immediately.
    """
    # 1. Insert Matches
    matches_added = 0
    for match in matches:
        # Check if match already exists to avoid duplicates
        # We check by ID, which is the primary key
        result = await db.execute(select(Match).where(Match.id_jogo == match.id_jogo))
        existing_match = result.scalars().first()
        
        if existing_match:
            continue # Skip if already exists
            
        db_match = Match(**match.dict())
        if not db_match.data_jogo:
            db_match.data_jogo = datetime.now()
        db.add(db_match)
        matches_added += 1
    
    await db.commit()
    
    if matches_added == 0:
        return {"status": "skipped", "message": "No new matches to add (all duplicates). Optimization skipped."}

    # 2. Trigger Optimization
    return await run_optimization(db)

async def run_optimization(db: AsyncSession):
    """
    Shared logic for optimization/training.
    """
    analyzer = FootballAnalyzer(db)
    
    # 1. Fetch all matches
    matches_df = await analyzer.get_all_matches()
    
    if matches_df.empty:
        return {"status": "error", "message": "No matches found to train on."}
        
    # 2. Engineer Features
    # Note: prepare_features returns features aligned with historical matches for training
    training_data = analyzer.feature_engineer.prepare_features(matches_df)
    
    # 3. Train Models
    train_metrics = analyzer.ml_engine.train(training_data)
    
    # 4. Log Metrics
    if train_metrics and "mae_home" in train_metrics:
        # Convert MAE to a score (e.g., 100 - MAE*10, clamped) just for visualization
        accuracy_score = max(0, 100 - (train_metrics['mae_home'] + train_metrics['mae_away']) * 20)
        
        metrics = ModelMetrics(
            accuracy_score=int(accuracy_score),
            total_predictions=train_metrics['samples'],
            model_version=f"xgb_v{datetime.now().strftime('%Y%m%d%H%M')}"
        )
        db.add(metrics)
        await db.commit()
    
    return {"status": "Optimization complete", "metrics": train_metrics}

@app.get("/optimize/")
async def optimize_model(db: AsyncSession = Depends(get_db)):
    """
    Trigger the self-optimization loop.
    This fetches all matches, engineers features, and retrains the XGBoost models.
    """
    return await run_optimization(db)

if __name__ == "__main__":
    import uvicorn
    uvicorn.run(app, host="0.0.0.0", port=8000)
