import asyncio
import random
import numpy as np
from datetime import datetime, timedelta
from app.database import engine, Base, SessionLocal
from app.models import Match

TEAMS = [
    "ARS", "AST", "BHA", "BOU", "BRE", "CHE", "CRY", "EVE", "FUL", "IPS",
    "LEI", "LIV", "MNC", "MNU", "NEW", "NOT", "SOU", "TOT", "WHU", "WOL"
]

async def init_db():
    async with engine.begin() as conn:
        await conn.run_sync(Base.metadata.drop_all)
        await conn.run_sync(Base.metadata.create_all)

async def populate():
    await init_db()
    
    async with SessionLocal() as session:
        matches = []
        start_date = datetime.now() - timedelta(days=30)
        
        for i in range(100000): # Generate 1000 matches
            home = random.choice(TEAMS)
            away = random.choice([t for t in TEAMS if t != home])
            
            # Simple bias: some teams are stronger
            if "ARS" in home or "AST" in home or "BHA" in home or "BOU" in home or "BRE" in home:
                home_strength = 1.4
            else:
                home_strength = 1.0
                
            if "ARS" in away or "AST" in away or "BHA" in away or "BOU" in away or "BRE" in away:
                away_strength = 1.2
            else:
                away_strength = 0.9
                
            if "LEI" in home or "LIV" in home or "MNC" in home or "MNU" in home or "NEW" in home:
                home_strength = 1.2
            else:
                home_strength = 0.9
                
            if "LEI" in away or "LIV" in away or "MNC" in away or "MNU" in away or "NEW" in away:
                away_strength = 1.4
            else:
                away_strength = 1.0
            
            # Generate score based on Poisson distribution (mimicking real data)
            # Home advantage usually gives a boost
            home_score = np.random.poisson(1.5 * home_strength)
            away_score = np.random.poisson(1.1 * away_strength)
            
            match_date = start_date + timedelta(minutes=i*2) # Every 2 mins
            
            matches.append(Match(
                time_casa=home,
                time_fora=away,
                placar_casa=home_score,
                placar_fora=away_score,
                data_jogo=match_date
            ))
            
            if len(matches) >= 100:
                session.add_all(matches)
                await session.commit()
                matches = [] # Reset list but keep loop going
                print(f"Inserted matches up to {i+1}...")

        if matches:
            session.add_all(matches)
            await session.commit()
        
        print("Database populated successfully!")

if __name__ == "__main__":
    loop = asyncio.get_event_loop()
    loop.run_until_complete(populate())
